#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <asm/termios.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include "../slip.h"
#include "../file.h"
#include "../remote.h"

#define DEV "/dev/serial0"

int ser_fd;

/*-----------------------------------------------------------------*/

void slip_tx(char *buf,int len)
{
	unsigned char a;
	int i;

	a=FEND; write(ser_fd,&a,1);
	for (i=0;i<len;i++) {
		a=buf[i];
		if (a==FEND) {
			a=FESC; write(ser_fd,&a,1);
			a=TFEND; write(ser_fd,&a,1);
			continue;
		} else if (a==FESC) {
			a=FESC; write(ser_fd,&a,1);
			a=TFESC; write(ser_fd,&a,1);
			continue;
		    } else write(ser_fd,&a,1);
	}
	a=FEND; write(ser_fd,&a,1);
}

int slip_rx(char *buf)
{
	unsigned char a;
	int i,n;
	static int st=0;
	
	if(!st) {
	    do {
		read(ser_fd,&a,1);
	    } while (a!=FEND);
	    st=1;
	}
	do {
		read(ser_fd,&a,1);
	} while (a==FEND);
	if (a==FESC) {
		read(ser_fd,&a,1);
		if (a==TFEND) a=FEND;
		else if(a==TFESC) a=FESC;
		     else { st=0; return 0;}
	}
	buf[0]=a;
	
	for(i=1;i<=MAXLEN;i++) {
		read(ser_fd,&a,1);
		if (a==FEND) return i;
		if (i==MAXLEN) {st=0; return 0;}
		if (a==FESC) { 
			read(ser_fd,&a,1);
			if (a==TFEND) a=FEND;
			else if(a==TFESC) a=FESC;
			     else { st=0; return 0;}
		}
		buf[i]=a;
	}
	st=0; return 0;
}

/*---------------------------------------------------------------*/
// Rutinas para adaptar el endian del PC (little endian) al
// 68000 del daVinci (big endian)

#ifdef HOST_BIG_ENDIAN
// Versiones tontas para unix con procesador big endian
// (SPARC, HP PA-RISC, 680XX)

extern inline unsigned int swapl(unsigned int dato) {return dato;}
extern inline unsigned short swapw(unsigned short dato) {return dato;}

#else
// Versiones para procesadores little endian
// (Intel x86, Alpha, otros muchos...)

extern inline unsigned int swapl(unsigned int dato)
{
	union {
		unsigned int i32;
		unsigned char c[4];
	} a;

	a.c[0]=(dato>>24)&0xff;
	a.c[1]=(dato>>16)&0xff;
	a.c[2]=(dato>>8)&0xff;
	a.c[3]=(dato)&0xff;

	return a.i32;
}

extern inline unsigned short swapw(unsigned short dato)
{
	union {
		unsigned short i16;
		unsigned char c[2];
	} a;

	a.c[0]=(dato>>8)&0xff;
	a.c[1]=(dato)&0xff;

	return a.i16;
}

#endif

/*-----------------------------------------------------------------*/

struct {
	char *ext;
	unsigned char tipo;
	unsigned char subt;
} fidt[]={
	"del",	0,		0,
	"bin",	BINARY_DATA,	1,
	"pic",	BINARY_CODE,	MC_PIC,
	"txt",	TEXT,		ASCII_RAW,
	"c",	TEXT,		PROGRAM_SOURCE_C,
	"h",	TEXT,		PROGRAM_SOURCE_C,
	"s",	TEXT,		PROGRAM_SOURCE_ASM,
	"asm",	TEXT,		PROGRAM_SOURCE_ASM,
	"1bpp",	GRAPHIC,	SCREEN_DUMP_1BPP,
	"2bpp",	GRAPHIC,	SCREEN_DUMP_2BPP,
	"4bpp",	GRAPHIC,	SCREEN_DUMP_4BPP,
        "dct",  GRAPHIC,        WINDOW_ZIP,
	"ag",	AGENDA,		1,
	NULL,	0,		0
};

char *mkname(int tipo,int subt,char *fn)
{
	static char name[256];
	char ext[8];
	struct stat stb;

	int i;
	for(i=1;fidt[i].tipo;i++) {
		if (fidt[i].tipo==tipo && fidt[i].subt==subt) break;
	}

	if (fidt[i].tipo) strcpy(ext,fidt[i].ext);
	else sprintf(ext,"%02x%02x",tipo,subt);

	sprintf(name,"%s.%s",fn,ext);

	for (i=2;;i++) {
		if(stat(name,&stb)==-1) return name;
		sprintf(name,"%s#%d.%s",fn,i,ext);
	}

	return name;
}

int mk_fh(char *fn,file_header *fp)
{
	char *p,*pp,ext[32];
	int i;

	// Busqueda de extensin
	p=fn;
	p+=strlen(fn);
	while(*p!='.' && p>fn) p--;
	if (p==fn) return -1;

	p++; strcpy(ext,p); p--;

	// Buscamos directorio (/) y copiamos nombre
	pp=fn;
	pp+=strlen(fn);
	while(*pp!='/' && pp>=fn) pp--;
	pp++;
	for(i=0;i<15;i++) {
		if (pp==p) break;
		fp->name[i]=*pp++;
	}
	fp->name[i]=0;

	// Busqueda de tipo
	for(i=0;fidt[i].ext;i++) {
		if (strcmp(ext,fidt[i].ext)==0) break;
	}
	if (fidt[i].ext==NULL) return -1;
	fp->type=fidt[i].tipo;
	fp->subtype=fidt[i].subt;
	return 0;
}

/*--------------------------------------------*/

void show_use()
{
	printf("\n\nUso de dvtrf:\n");
	printf("-------------------------------------------\n");
	printf("dvtrf get tipo subtipo  (tipo=0 => todos los ficheros)\n");
	printf("dvtrf put file_1 .... file_n\n");
	printf("dvtrf dump file.pgm\n\n");
}

/*------------------------------------------------------------------*/

void get_files(int argc,char **argv)
{
	int i,j,k,np,sz,fd,tipo,subt,nf;
	unsigned int size,rtipo,rsubt;
	char *fname;
	char buft[MAXLEN],bufr[MAXLEN];
	unsigned short 	*ps;
	unsigned int	*pi;
	file_header	*pf;
	pkt_get_nfiles 	*pgnf;
	pkt_open_file	*pof;

	if (argc<4) {show_use(); exit(0);}
	tipo=atoi(argv[2]);
	subt=atoi(argv[3]);

	pgnf=(pkt_get_nfiles *)buft;
	pgnf->code=swapw(RMT_GET_NFILES);
	pgnf->type=tipo;
	pgnf->subtype=subt;
	slip_tx(buft,sizeof(pkt_get_nfiles));
	i=slip_rx(bufr);
	ps=(unsigned short *)bufr;
	if (i<6 || *ps==swapw(RMT_ERROR)) {printf("\nError\n"); exit(1); }
	pi=(unsigned int *)++ps;
	nf=swapl(*pi);
	printf("%d Ficheros\n",nf);
	for(i=0;i<nf;i++) {
		pof=(pkt_open_file *)buft;
		pof->code=swapw(RMT_OPEN_FILE);
		pof->type=tipo;
		pof->subtype=subt;
		pof->index=swapl(i);
		slip_tx(buft,sizeof(pkt_open_file));
		j=slip_rx(bufr);
		ps=(unsigned short *)bufr;
		if (j<2+sizeof(file_header) || *ps==swapw(RMT_ERROR)) {
			printf("\nError\n"); exit(1); }
		pf=(file_header *)++ps;
		size=swapl(pf->size);
		rtipo=pf->type;
		rsubt=pf->subtype;
		fname=mkname(rtipo,rsubt,pf->name);
		printf("%3d %3d %6d %s       ",rtipo,rsubt,size,fname);
		if ((fd=creat(fname,0666))==-1) {perror("creat"); exit(1);}
		sz=size;

                // Usamos Full-duplex para establecer un pipeline
                // Mientras leeemos un bloque vamos pidiendo el siguiente
                // Resultado: 11 KByte/seg bajando ficheros.
                np=0;
		do {
			ps=(unsigned short *)buft;
			*ps=swapw(RMT_READ_FILE);
			slip_tx(buft,2);
                        if (np) {  // No empezamos a leer hasta el 2 bloque
			  j=slip_rx(bufr);
			  ps=(unsigned short *)bufr;
			  if(j<2 || *ps!=swapw(RMT_ACK)) {
				printf("\nError\n"); exit(1);}
			  j-=2;
			  write(fd,(char *)++ps,j);
			  sz-=j;
                        }
			k=100-(sz*100)/size;
			printf("\b\b\b\b\b%4d%%",k); fflush(stdout);
                        np++;
		} while (sz>0);
                // Queda por leer un bloque
                j=slip_rx(bufr);
		ps=(unsigned short *)bufr;
		if(j<2 || *ps!=swapw(RMT_ACK)) {
			printf("\nError\n"); exit(1);}
		j-=2;
		write(fd,(char *)++ps,j);
		sz-=j;

                printf("\n");
		close(fd);
	}

}

/*---------------------------------------------------------------*/

void put_files(int argc,char **argv)
{
	int i,j,k,np,size,size0,fd,nf;
	unsigned short *ps;
	file_header *pf;
	char buft[MAXLEN],bufr[MAXLEN];

	if (argc <3) {show_use(); exit(0);}

	for (nf=2;nf<argc;nf++) {
		ps=(unsigned short *)buft;
		*ps++=swapw(RMT_CREAT_FILE);
		pf=(file_header *)ps;
		if (mk_fh(argv[nf],pf)==-1) {
			printf("%s descartado\n",argv[nf]);
			continue;
		};
		if((fd=open(argv[nf],O_RDONLY))==-1) {
			printf("%s descartado\n",argv[nf]);
			continue;
		}
		pf->magic=swapl(FILE_MAGIC);
		size=size0=lseek(fd,0,SEEK_END);
		lseek(fd,0,SEEK_SET);
		pf->size=swapl(size);

		printf("%d %d %d %s     ",pf->type,pf->subtype,size,pf->name);
		fflush(stdout);

		slip_tx(buft,sizeof(pkt_creat_file));
		i=slip_rx(bufr);
		ps=(unsigned short *)bufr;
		if (i<2 || *ps!=swapw(RMT_ACK)) {
			printf("\nError en creat\n");
			exit(1);
		}
                // Usamos Full-duplex para establecer un pipeline
                // Mientras el davinci programa un bloque en la flash
                // vamos enviando el siguiente
                // Resultado: 10 KByte/seg subiendo ficheros.
                np=0;
		do {
			ps=(unsigned short *)buft;
			*ps++=swapw(RMT_WRITE_FILE);
			i=read(fd,(char *)ps,MAXLEN-2);
			slip_tx(buft,i+2);
                        // No esperamos ACK hasta el segundo bloque
                        if (np) {
                          j=slip_rx(bufr);
			  ps=(unsigned short *)bufr;
			  if (j<2 || *ps!=swapw(RMT_ACK)) {
				printf("\nError en write\n");
				exit(1);
			  }
                        }
			size-=i;
			k=100-(100*size)/size0;
			printf("\b\b\b\b\b%4d%%",k); fflush(stdout);
                        np++;
		} while (size>0);
                // Ultimo bloque
                j=slip_rx(bufr);
		ps=(unsigned short *)bufr;
		if (j<2 || *ps!=swapw(RMT_ACK)) {
			printf("\nError en write\n");
			exit(1);
		}

		close(fd);
		printf("\n");
	}
}

/*----------------------------------------------------------------------*/

void dump_pgm(char *pp,int modo,FILE *fp)
{
	int i,j;
	unsigned short *p,a,b;
	char tb2b[4]={0,85,170,255};
	char tb4b[16]={	0,17,34,51,68,85,102,119,136,153,170,187,
			204,221,238,255};
	
	fprintf(fp,"P5\n");
	fprintf(fp,"#CREATOR daVinci FEOS\n");
	fprintf(fp,"#version 2.2\n");
	fprintf(fp,"160 160\n");
	fprintf(fp,"255\n");

	p=(unsigned short *)pp;
	
	switch(modo) {
	case 0:	// 1 bit/pixel
		for (i=0;i<1600;i++){
			a=swapw(~*p++);
			for (j=0;j<16;j++) {
				if (a&0x8000) putc(0xff,fp);
				else putc(0,fp);
				a<<=1;
			}
		}
		break;
	case 1:	// 2 bit/pixel
		for (i=0;i<3200;i++){
			a=swapw(~*p++);
			for (j=0;j<8;j++) {
				b=(a&0xc000)>>14;
				putc(tb2b[b],fp);
				a<<=2;
			}
		}
		break;
	case 2:	// 4 bit/pixel
		for (i=0;i<6400;i++){
			a=swapw(~*p++);
			for (j=0;j<4;j++) {
				b=(a&0xF000)>>12;
				putc(tb4b[b],fp);
				a<<=4;
			}
		}
		break;
	}
}

void dump_screen(int argc,char **argv)
{
	int i,j,k,np,modo;
	int size;
	char screen[12800],*p;
	char buft[MAXLEN],bufr[MAXLEN];
	FILE *fp;
	unsigned short 	*ps;

	if (argc<3) {show_use(); exit(0);}

	ps=(unsigned short *)buft;
	*ps=swapw(RMT_OPEN_SCREEN);
	slip_tx(buft,2);
	i=slip_rx(bufr);
	ps=(unsigned short *)bufr;
	if(i<2 || *ps!=swapw(RMT_ACK)) {
		printf("\nError open_screen\n");
		exit(1);
	}
	p=screen;
	printf("Capturando pantalla"); fflush(stdout);
        np=0;
	do {
		ps=(unsigned short *)buft;
		*ps=swapw(RMT_READ_FILE);
		slip_tx(buft,2);
		if (np) {
                  j=slip_rx(bufr);
		  ps=(unsigned short *)bufr;
		  if(j<2 || *ps!=swapw(RMT_ACK)) {
			printf("\nError\n"); exit(1);}
		  j-=2;
		  memcpy(p,(char *)++ps,j);
		  p+=j;
                }
		putchar('.'); fflush(stdout);
                np++;
	} while (j);
        j=slip_rx(bufr);
	printf("\n");

	size=(int)p-(int)screen;
	if (size==12800) modo=2;
	else if(size==6400) modo=1;
	     else modo=0;

	if ((fp=fopen(argv[2],"w"))==NULL) {
		perror("fopen");
		exit(1);
	}
	printf("Generando PGM\n");
	dump_pgm(screen,modo,fp);
	fclose(fp);
}
/*----------------------------------------------------------------------*/

main(int argc,char **argv)
{
	unsigned char a,buft[MAXLEN],bufr[MAXLEN];
	int i,j,cid,baud=0,result;
	unsigned short *ps;
	struct termios term;
	char rotor[4]={'|','/','-','\\'};
	
	if (argc<3) {show_use(); exit(0);}
	
	if ((ser_fd=open(DEV,O_RDWR))==-1) exit(0);
	baud=B115200;
			
	/* ponemos las cosas del puerto en su sitio */
	tcgetattr(ser_fd,&term)
	;
	term.c_cflag&=~CBAUD;
	term.c_cflag|=baud;
	term.c_cflag&=~CSIZE;
	term.c_cflag|=CS8;
	term.c_cflag|=CSTOPB;
	term.c_cflag&=~PARENB;
	
	term.c_lflag&=~ICANON;
	term.c_lflag&=~ECHO;
	term.c_lflag&=~ISIG;
	term.c_iflag&=~ICRNL;
	term.c_iflag&=~IXOFF;
	term.c_iflag&=~IXON; 
	term.c_oflag&=~OPOST;

	
	if (tcsetattr(ser_fd,TCSANOW,&term)) {
		fprintf(stderr,"Fallo en tcsetattr\n");
		exit(0);
	}

	/* Damos alimentacin al cradle */

	result=0;
	result|=TIOCM_DTR;
	ioctl (ser_fd, TIOCMBIS, &result);

	result=0;
	result|=TIOCM_RTS;
	ioctl (ser_fd, TIOCMBIC, &result);
	
	/* Esperamos el establecimiento de la conexin */
	printf("Estableciendo conexin ...  "); fflush(stdout);
	if (!(cid=fork())) {
		/* proceso hijo */
		ps=(unsigned short *)buft;
		*ps++=swapw(RMT_ECHO);
		for (i=0;;i++) {
			printf("\b%c",rotor[i&3]); fflush(stdout);
			slip_tx(buft,2);
			sleep(1);
		}
	}
	/* proceso padre */
	
	ps=(unsigned short *)bufr;
	for(;;) {
		i=slip_rx(bufr);
		if(i==2 && *ps==swapw(RMT_ECHO)) break;
	}
	kill(cid,SIGKILL);
	printf("\bOK\n");
	
	if (strcmp(argv[1],"get")==0) get_files(argc,argv);
	if (strcmp(argv[1],"put")==0) put_files(argc,argv);
	if (strcmp(argv[1],"dump")==0) dump_screen(argc,argv);
}

