#include "common.h"
#include "mem.h"
#include "task.h"
#include "lcd.h"
#include "klib.h"
#include "event.h"
#include "lang.h"

extern TSS *cur_task;

struct {
	unsigned char *buffer_pointer;	//+0
	unsigned char *buffer_base;	//+4
	unsigned char *buffer_end;	//+8
	unsigned short time_rise;	//+12
	unsigned int acc_period;	//+14
	unsigned int ncycles;		//+18
} audio_st;

struct {
	unsigned char *buffer_pointer;  //+0
	unsigned char *buffer_base;     //+4
	unsigned char *buffer_end;      //+8
} pwm_st;

/*-------------- Rutina de interrupcin de nivel 6 ----------------*/

asm("
	.globl l6_isr
	.globl sys_audio
	.globl sys_pwm
l6_isr:
	btst.b	#3,0xfffff30d	|Causa de la interrupcin IRQ6
	bne	1f
	btst.b	#7,0xfffff30f	|Causa de la interrupcin PWMO
	bne	4f
	bra	l6_isr_timer
1:	movm.l	%d0-%d1/%a0-%a1,-(%sp)
	lea.l	audio_st,%a0
	bchg.b	#4,0xfffff302	|Cambiamos y probamos polaridad
	beq	2f
	bset.b	#3,0xfffff30d	|Borramos interrupcin
	move.w	0xfffff608,%d0	|Flanco de subida
	move.w	12(%a0),%d1
	move.w	%d0,12(%a0)
	sub.w	%d1,%d0		|duracin de 1 ciclo
	andi.l	#0xffff,%d0
	add.l	%d0,14(%a0)	|acumulamos duraciones de ciclos
	addq.l	#1,18(%a0)	|incrementamos contador de ciclos
	movm.l	(%sp)+,%d0-%d1/%a0-%a1
	rte
2:	bset.b	#3,0xfffff30d	|Borramos interrupcin
	move.w	0xfffff608,%d0	|Flanco de bajada
	sub.w	12(%a0),%d0	|ancho del pulso
	movea.l	(%a0),%a1
	move.b	%d0,(%a1)+	|Al buffer
	cmpa.l	8(%a0),%a1
	bne	3f
	movea.l	4(%a0),%a1
3:	move.l	%a1,(%a0)
	movm.l  (%sp)+,%d0-%d1/%a0-%a1
	rte

|-------------------- PWMO --------------------

4:	tst.w	0xfffff500	|para borrar IRQ
	movm.l	%a0-%a1,-(%sp)
	lea.l	pwm_st,%a0
	movea.l	(%a0),%a1
	move.w	(%a1)+,0xfffff502
	cmpa.l	8(%a0),%a1
	bne	5f
	movea.l	4(%a0),%a1
5:	move.w	(%a1)+,0xfffff502
	cmpa.l	8(%a0),%a1
	bne	6f
	movea.l	4(%a0),%a1
6:	move.l	%a1,(%a0)
	movm.l	(%sp)+,%a0-%a1
	rte


|-----------------------------------------------------------------------

sys_audio:	| Llamada rpida. retorna el puntero del buffer circular
	move.l	audio_st,%d0
	rte

|-----------------------------------------------------------------------

sys_pwm:	|Llamada rpida. retorna el puntero del buffer circular
	move.l	pwm_st,%d0
	rte
");

/*---------------------------------------------------------------------*/

TSS *audio_task;

int audio_start(unsigned char *buf,int len)
{
	static char busy=0;

	lock(&busy);

	if (audio_task) {unlock(&busy); return -1;}

	if (check_mem_range(buf,len)) {unlock(&busy); return -1;}
	audio_st.buffer_pointer=audio_st.buffer_base=buf;
	audio_st.buffer_end=&buf[len];
	audio_st.acc_period=audio_st.ncycles=0;

	*(volatile unsigned short *)0xfffff302|=0x1100;	//IRQ6 edge MODE
	asm volatile("
	bclr.b	#3,0xfffff305	|IRQ6 ACTIVA
	bset.b	#3,0xfffff30d	|Borramos IRQ6
	");

	audio_task=cur_task;
	unlock(&busy);
	return 0;
}

void audio_stop()
{
	asm volatile(" bset.b	#3,0xfffff305	|IRQ6 inactiva");
	audio_task=(TSS *)0;
}

int audio_mide_frec()
{
	unsigned int i,j,k;
	unsigned short flg;

	for (i=0;i<16;i++) {
            if (audio_st.ncycles==0) task_sleep(8); else break;
        }
        if (i==16) return 0;

	asm volatile ("
		move.w %%sr,%0
		move.w #0x2700,%%sr
	":"=d"(flg):);
	j=audio_st.ncycles;
	k=audio_st.acc_period;
	asm volatile ("move.w %0,%%sr"::"d"(flg));

	task_sleep(8);

	asm volatile ("
		move.w %%sr,%0
		move.w #0x2700,%%sr
	":"=d"(flg):);
	j=audio_st.ncycles-j;
	k=audio_st.acc_period-k;
	asm volatile ("move.w %0,%%sr"::"d"(flg));
	if (k) i=(2072576*j)/k; else i=0;
	return i;
}

/*--------------------------------------------------------------------------*/

void gettap(vt*);

void tunning_tool()
{
	vt *pvt;
	char *sbase;
	int f;

	pvt=cur_task->vt;
	if (!pvt) return;
	sbase=pvt->screen_base;

	cls();
	kprintf("Buffer:\n\n\n");

	if (audio_start(&sbase[200],160)) {
		kprintf(MSG_AUDIO_BUSY);
		gettap(pvt);
		return;
	}
	gotoxy(0,18);

	do {
		f=audio_mide_frec();
		kprintf("\rFrec: %5d Hz",f);
		task_sleep(16);
		f=fifo_getnb(pvt->event);
	} while (f!=EV_LCD_TAP_PRESS);

	audio_stop();
}

/*****************************************************/
/********************** PWMO *************************/
/*****************************************************/

TSS *pwm_task;
extern char pwm_busy;
int tas(char *busy);

int pwm_start(unsigned char *buf,int len)
{
	if ((int)buf & 1) return -1;	// Alineacin PAR
	if (len & 1) return -1;		// Tamao PAR
	if (check_mem_ro(buf,len)) return -1;	// No BUS_ERROR

	if (!tas(&pwm_busy)) return -1;	// PWM ocupado

	pwm_task=cur_task;
	pwm_st.buffer_pointer=pwm_st.buffer_base=buf;
	pwm_st.buffer_end=&buf[len];

	asm volatile("bclr.b #7,0xfffff307");	// HAbilitando IRQ
	*(unsigned short *)0xfffff500=0x0058;	// IRQ, PWM enable REPEATx4
	*(unsigned char  *)0xfffff504=0xfe;	// Periodo=256
	*(unsigned short *)0xfffff502=0x8080;	// Datos iniciales para FIFO
	*(unsigned short *)0xfffff502=0x8080;

	return 0;
}

void pwm_stop()
{
	*(unsigned short *)0xfffff500=0;	// PWM deshabilitado
	asm volatile("bset.b #7,0xfffff307");	// Deshabilitamos IRQ
	pwm_task=(TSS *)0;

	unlock(&pwm_busy);
}
